/*
 * Copyright 1998,1999 The OpenLDAP Foundation, Redwood City, California, USA
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted only
 * as authorized by the OpenLDAP Public License.  A copy of this
 * license is available at http://www.OpenLDAP.org/license.html or
 * in file LICENSE in the top-level directory of the distribution.
 */
/* Portions
 * Copyright (c) 1993, 1994 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 * disptmpl.h:  display template library defines
 * 7 March 1994 by Mark C Smith
 */

#ifndef _DISPTMPL_H
#define _DISPTMPL_H

#include <ldap_cdefs.h>

LDAP_BEGIN_DECL

#define LDAP_TEMPLATE_VERSION	1

/*
 * general types of items (confined to most significant byte)
 */
#define LDAP_SYN_TYPE_TEXT		0x01000000L
#define LDAP_SYN_TYPE_IMAGE		0x02000000L
#define LDAP_SYN_TYPE_BOOLEAN		0x04000000L
#define LDAP_SYN_TYPE_BUTTON		0x08000000L
#define LDAP_SYN_TYPE_ACTION		0x10000000L


/*
 * syntax options (confined to second most significant byte)
 */
#define LDAP_SYN_OPT_DEFER		0x00010000L


/* 
 * display template item syntax ids (defined by common agreement)
 * these are the valid values for the ti_syntaxid of the tmplitem
 * struct (defined below).  A general type is encoded in the
 * most-significant 8 bits, and some options are encoded in the next
 * 8 bits.  The lower 16 bits are reserved for the distinct types.
 */
#define LDAP_SYN_CASEIGNORESTR	( 1 | LDAP_SYN_TYPE_TEXT )
#define LDAP_SYN_MULTILINESTR	( 2 | LDAP_SYN_TYPE_TEXT )
#define LDAP_SYN_DN		( 3 | LDAP_SYN_TYPE_TEXT )
#define LDAP_SYN_BOOLEAN	( 4 | LDAP_SYN_TYPE_BOOLEAN )
#define LDAP_SYN_JPEGIMAGE	( 5 | LDAP_SYN_TYPE_IMAGE )
#define LDAP_SYN_JPEGBUTTON	( 6 | LDAP_SYN_TYPE_BUTTON | LDAP_SYN_OPT_DEFER )
#define LDAP_SYN_FAXIMAGE	( 7 | LDAP_SYN_TYPE_IMAGE )
#define LDAP_SYN_FAXBUTTON	( 8 | LDAP_SYN_TYPE_BUTTON | LDAP_SYN_OPT_DEFER )
#define LDAP_SYN_AUDIOBUTTON	( 9 | LDAP_SYN_TYPE_BUTTON | LDAP_SYN_OPT_DEFER )
#define LDAP_SYN_TIME		( 10 | LDAP_SYN_TYPE_TEXT )
#define LDAP_SYN_DATE		( 11 | LDAP_SYN_TYPE_TEXT )
#define LDAP_SYN_LABELEDURL	( 12 | LDAP_SYN_TYPE_TEXT )
#define LDAP_SYN_SEARCHACTION	( 13 | LDAP_SYN_TYPE_ACTION )
#define LDAP_SYN_LINKACTION	( 14 | LDAP_SYN_TYPE_ACTION )
#define LDAP_SYN_ADDDNACTION	( 15 | LDAP_SYN_TYPE_ACTION )
#define LDAP_SYN_VERIFYDNACTION ( 16 | LDAP_SYN_TYPE_ACTION )
#define LDAP_SYN_RFC822ADDR	( 17 | LDAP_SYN_TYPE_TEXT )


/*
 * handy macros
 */
#define LDAP_GET_SYN_TYPE( syid )	((syid) & 0xFF000000L )
#define LDAP_GET_SYN_OPTIONS( syid )	((syid) & 0x00FF0000L )


/*
 * display options for output routines (used by entry2text and friends)
 */
/*
 * use calculated label width (based on length of longest label in
 * template) instead of contant width
 */
#define LDAP_DISP_OPT_AUTOLABELWIDTH	0x00000001L
#define LDAP_DISP_OPT_HTMLBODYONLY	0x00000002L

/*
 * perform search actions (applies to ldap_entry2text_search only) 
 */
#define LDAP_DISP_OPT_DOSEARCHACTIONS	0x00000002L

/*
 * include additional info. relevant to "non leaf" entries only
 * used by ldap_entry2html and ldap_entry2html_search to include "Browse"
 * and "Move Up" HREFs
 */
#define LDAP_DISP_OPT_NONLEAF		0x00000004L


/*
 * display template item options (may not apply to all types)
 * if this bit is set in ti_options, it applies.
 */
#define LDAP_DITEM_OPT_READONLY		0x00000001L
#define LDAP_DITEM_OPT_SORTVALUES	0x00000002L
#define LDAP_DITEM_OPT_SINGLEVALUED	0x00000004L
#define LDAP_DITEM_OPT_HIDEIFEMPTY	0x00000008L
#define LDAP_DITEM_OPT_VALUEREQUIRED	0x00000010L
#define LDAP_DITEM_OPT_HIDEIFFALSE	0x00000020L	/* booleans only */



/*
 * display template item structure
 */
struct ldap_tmplitem {
    unsigned long		ti_syntaxid;
    unsigned long		ti_options;
    char  			*ti_attrname;
    char			*ti_label;
    char			**ti_args;
    struct ldap_tmplitem	*ti_next_in_row;
    struct ldap_tmplitem	*ti_next_in_col;
    void			*ti_appdata;
};


#define LDAP_SET_TMPLITEM_APPDATA( ti, datap )	\
	( (ti)->ti_appdata = (void *)(datap) )

#define LDAP_GET_TMPLITEM_APPDATA( ti, type )	\
	( (type)((ti)->ti_appdata) )

#define LDAP_IS_TMPLITEM_OPTION_SET( ti, option )	\
	( ((ti)->ti_options & (option) ) != 0 )


/*
 * object class array structure
 */
struct ldap_oclist {
    char		**oc_objclasses;
    struct ldap_oclist	*oc_next;
};


/*
 * add defaults list
 */
struct ldap_adddeflist {
    int			ad_source;
#define LDAP_ADSRC_CONSTANTVALUE	1
#define LDAP_ADSRC_ADDERSDN		2
    char		*ad_attrname;
    char		*ad_value;
    struct ldap_adddeflist	*ad_next;
};


/*
 * display template global options
 * if this bit is set in dt_options, it applies.
 */
/*
 * users should be allowed to try to add objects of these entries
 */
#define LDAP_DTMPL_OPT_ADDABLE		0x00000001L

/*
 * users should be allowed to do "modify RDN" operation of these entries
 */
#define LDAP_DTMPL_OPT_ALLOWMODRDN	0x00000002L

/*
 * this template is an alternate view, not a primary view
 */
#define LDAP_DTMPL_OPT_ALTVIEW		0x00000004L


/*
 * display template structure
 */
struct ldap_disptmpl {
    char			*dt_name;
    char			*dt_pluralname;
    char			*dt_iconname;
    unsigned long		dt_options;
    char			*dt_authattrname;
    char			*dt_defrdnattrname;
    char			*dt_defaddlocation;
    struct ldap_oclist		*dt_oclist;
    struct ldap_adddeflist	*dt_adddeflist;
    struct ldap_tmplitem	*dt_items;
    void			*dt_appdata;
    struct ldap_disptmpl	*dt_next;
};

#define LDAP_SET_DISPTMPL_APPDATA( dt, datap )	\
	( (dt)->dt_appdata = (void *)(datap) )

#define LDAP_GET_DISPTMPL_APPDATA( dt, type )	\
	( (type)((dt)->dt_appdata) )

#define LDAP_IS_DISPTMPL_OPTION_SET( dt, option )	\
	( ((dt)->dt_options & (option) ) != 0 )

#define LDAP_TMPL_ERR_VERSION	1
#define LDAP_TMPL_ERR_MEM	2
#define LDAP_TMPL_ERR_SYNTAX	3
#define LDAP_TMPL_ERR_FILE	4

/*
 * buffer size needed for entry2text and vals2text
 */
#define LDAP_DTMPL_BUFSIZ	8192


typedef int (*ldap_writeptype) LDAP_P((
	void *writeparm, char *p, ber_len_t len ));

LDAP_F( int )
ldap_init_templates LDAP_P(( char *file, struct ldap_disptmpl **tmpllistp ));

LDAP_F( int )
ldap_init_templates_buf LDAP_P(( char *buf,
	ber_len_t buflen,
	struct ldap_disptmpl **tmpllistp ));

LDAP_F( void )
ldap_free_templates LDAP_P(( struct ldap_disptmpl *tmpllist ));

LDAP_F( struct ldap_disptmpl * )
ldap_first_disptmpl LDAP_P(( struct ldap_disptmpl *tmpllist ));

LDAP_F( struct ldap_disptmpl * )
ldap_next_disptmpl LDAP_P(( struct ldap_disptmpl *tmpllist,
	struct ldap_disptmpl *tmpl ));

LDAP_F( struct ldap_disptmpl * )
ldap_name2template LDAP_P(( char *name,
	struct ldap_disptmpl *tmpllist ));

LDAP_F( struct ldap_disptmpl * )
ldap_oc2template LDAP_P(( char **oclist,
	struct ldap_disptmpl *tmpllist ));

LDAP_F( char ** )
ldap_tmplattrs LDAP_P(( struct ldap_disptmpl *tmpl,
	char **includeattrs,
	int exclude,
	unsigned long syntaxmask ));

LDAP_F( struct ldap_tmplitem * )
ldap_first_tmplrow LDAP_P(( struct ldap_disptmpl *tmpl ));

LDAP_F( struct ldap_tmplitem * )
ldap_next_tmplrow LDAP_P(( struct ldap_disptmpl *tmpl,
	struct ldap_tmplitem *row ));

LDAP_F( struct ldap_tmplitem * )
ldap_first_tmplcol LDAP_P(( struct ldap_disptmpl *tmpl,
	struct ldap_tmplitem *row ));

LDAP_F( struct ldap_tmplitem * )
ldap_next_tmplcol LDAP_P(( struct ldap_disptmpl *tmpl,
	struct ldap_tmplitem *row,
	struct ldap_tmplitem *col ));

LDAP_F( int )
ldap_entry2text LDAP_P(( LDAP *ld,
	char *buf, LDAPMessage *entry,
	struct ldap_disptmpl *tmpl, char **defattrs, char ***defvals,
	ldap_writeptype writeproc, void *writeparm, char *eol, int rdncount,
	unsigned long opts ));

LDAP_F( int )
ldap_vals2text LDAP_P(( LDAP *ld,
	char *buf, char **vals, char *label, int labelwidth,
	unsigned long syntaxid, ldap_writeptype writeproc, void *writeparm,
	char *eol, int rdncount ));

LDAP_F( int )
ldap_entry2text_search LDAP_P(( LDAP *ld,
	char *dn, char *base, LDAPMessage *entry,
	struct ldap_disptmpl *tmpllist, char **defattrs, char ***defvals,
	ldap_writeptype writeproc, void *writeparm, char *eol, int rdncount,
	unsigned long opts ));

LDAP_F( int )
ldap_entry2html LDAP_P(( LDAP *ld,
	char *buf, LDAPMessage *entry,
	struct ldap_disptmpl *tmpl, char **defattrs, char ***defvals,
	ldap_writeptype writeproc, void *writeparm, char *eol, int rdncount,
	unsigned long opts, char *urlprefix, char *base ));

LDAP_F( int )
ldap_vals2html LDAP_P(( LDAP *ld,
	char *buf, char **vals, char *label, int labelwidth,
	unsigned long syntaxid, ldap_writeptype writeproc, void *writeparm,
	char *eol, int rdncount, char *urlprefix ));

LDAP_F( int )
ldap_entry2html_search LDAP_P(( LDAP
	*ld, char *dn, char *base, LDAPMessage *entry,
	struct ldap_disptmpl *tmpllist, char **defattrs, char ***defvals,
	ldap_writeptype writeproc, void *writeparm, char *eol, int rdncount,
	unsigned long opts, char *urlprefix ));

LDAP_END_DECL
#endif /* _DISPTMPL_H */
